/* Copyright (c) 2009, Oracle and/or its affiliates. All rights reserved. */

/*
 *********************** UNIT TEST INTERFACE ***************************

  NAME
  clsagfwm.h - Macros for inclusion in client agents, to test 
  functionality independent of Cluster Ready Service.

  DESCRIPTION
  The macros defined in this file help crs agent developers to add test 
  types and test resources. This testing can be done independent of 
  the presence of Clusterware components. Once the testing is successful,
  the agent and its associated resources can be integrated into Clusterware
  for automatic monitoring.

  RELATED DOCUMENTS
  (1) clsagfw.h - Documents the API for the Agent Framework
  (2) macrodemo.cpp - Example agent that uses the test macros functionality.
      This file should be present in your installation under the crs/demo
      directory.
  (3) Oracle Clusterware Administration and Deployment Guide.

  EXPORT MACROS
  The following is a list of macros defined in this file. Detailed comments 
  and descriptions can be found at their points of definition.

     AGFW_BEGINTEST_ARRAY    - Begin marker for array of unit tests.
     AGFW_ENDTEST_ARRAY      - End marker for array of unit tests.
     AGFW_ADDTEST            - Add a test action.
     AGFW_BEGINTEST_ACTIONS  - Begin marker for single test.
     AGFW_ENDTEST_ACTIONS    - End marker for single test.
     AGFW_ADDTYPE_BEGIN      - Begin marker to define a resource type.
     AGFW_ADD_INTATTR        - Add an integer attribute.
     AGFW_ADD_STRATTR        - Add a string attribute.
     AGFW_ADDTYPE_END        - Add a resource type definition.
     AGFW_ENV_SEPARATOR      - End marker to define a resource type.
     AGFW_ADDRES_BEGIN       - Begin marker to add a resource type.
     AGFW_ADDRES_END         - End marker to add a resource type.
     AGFW_MODIFY_INTATTR     - Modify an integer attribute.
     AGFW_MODIFY_STRATTR     - Modiy a string attibute.
     AGFW_START_RESOURCE     - Start a resource.
     AGFW_STOP_RESOURCE      - Stop a resource.
     AGFW_CLEAN_RESOURCE     - Clean a resource.
     AGFW_FAULT_RESOURCE     - Fail a resource.
     AGFW_SLEEP_SECS         - Sleep for specified time.
     AGFW_CHECK_THRDS        - Check thread status.

  EXAMPLE
  The following text can be added directly to your agent source code, to add
  a "testType" type with an integer and string attribute, add a resource of 
  type testType, etc.

  AGFW_BEGINTEST_ACTIONS(g_test_1_actions)
      AGFW_ADDTYPE_BEGIN("testType"),
      AGFW_ADD_INTATTR("attr1", "30"),
      AGFW_ADD_STRATTR("attr2", "attrvalue"),
      AGFW_ADDTYPE_END(),

      AGFW_ADDRES_BEGIN("testType", "testResource"),
      AGFW_MODIFY_INTATTR("checkInterval", "20"),
      AGFW_MODIFY_STRATTR("attr2", "newattrvalue"),
      AGFW_ADDRES_END(),

      AGFW_START_RESOURCE("testResource1"),
      AGFW_STOP_RESOURCE("testResource1"),
      AGFW_CLEAN_RESOURCE("testResource1"),
      AGFW_START_RESOURCE("testResource1"),
      AGFW_FAULT_RESOURCE("testResource1"),
  AGFW_ENDTEST_ACTIONS()

  AGFW_BEGINTEST_ARRAY(g_test_array)  
      AGFW_ADDTEST("test_1", (clsagfw_testentry *)g_test_1_actions)
      AGFW_ADDTEST("test_2", (clsagfw_testentry *)g_test_2_actions)
      AGFW_ADDTEST("test_3", (clsagfw_testentry *)g_test_3_actions)
  AGFW_ENDTEST_ARRAY()

*/

#ifndef CLSAGFW_MACROS
#define CLSAGFW_MACROS

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

#ifndef ORATYPES
# include <oratypes.h>
#endif

/*
 * Struct used internally to define a test operation
 */
typedef struct 
{
    sb4     testOperation;
    char    *testObj1;
    char    *testObj2;
}clsagfw_testentry;


/*
 * Used internally to define the array of test structures.
 */
typedef struct clsagfw_testarray
{
     oratext            *pTestName;
     clsagfw_testentry  *pTestEntries;
} clsagfw_testarray;


/**
 * Agent developers can use this macro to add unit test entry table to
 * the agent
 */

typedef enum
{
    AGFW_ADDTYPE_BEGIN          = 0x1,
    AGFW_ADD_INTATTR            = 0x2,
    AGFW_ADD_STRATTR            = 0x3,
    AGFW_ADDTYPE_END            = 0x4,

    AGFW_ADDRES_BEGIN           = 0x101,
    AGFW_MODIFY_INTATTR         = 0x102,
    AGFW_MODIFY_STRATTR         = 0x103,
    AGFW_ADDRES_END             = 0x104,

    AGFW_START_RESOURCE         = 0x1001,
    AGFW_STOP_RESOURCE          = 0x1002,
    AGFW_FAULT_RESOURCE         = 0x1003,
    AGFW_CLEAN_RESOURCE         = 0x1007,

    AGFW_SLEEP_SECS             = 0x1004,
    AGFW_CHECK_THRDS            = 0x1005,
    AGFW_TIMER_TEST             = 0x1006,

    AGFW_TESTEND_MARKER         = 0x0000
}clsagfw_test_opcode;



/*
 * Unit test start and end markers. All the test actions has to be
 * between these two markers.
 */
#define AGFW_BEGINTEST_ARRAY(test_array) \
  const clsagfw_testarray test_array[] = {
#define AGFW_ENDTEST_ARRAY() {(oratext *)AGFW_TESTEND_MARKER, NULL}};

/*
 * Add a test name and corresponding test actions.
 */
#define AGFW_ADDTEST(name, actions) {(name), (actions)},


/*
 * Unit test start and end markers. All the test actions has to be
 * between these two markers.
 */
#define AGFW_BEGINTEST_ACTIONS(actions_table) \
  const clsagfw_testentry actions_table[] = {
#define AGFW_ENDTEST_ACTIONS() {AGFW_TESTEND_MARKER, NULL, NULL}};


/*
 * Adds a resource type to the agfw unit test.
 *
 * Args:
 * AGFW_ADDTYPE_BEGIN(a) a = typename.
 *
 * AGFW_ADD_INTATTR(a, b) a = attrname, b = integer attr value in
 * string format.
 *
 * AGFW_ADD_STRATTR(a, b) a = attrname, b = attr value in string
 * format.
 *
 * AGFW_ADDTYPE_END()  End marker for type creation.
 *
 */
#define AGFW_ADDTYPE_BEGIN(a)      {AGFW_ADDTYPE_BEGIN, (a), NULL},
#define AGFW_ADD_INTATTR(a, b)     {AGFW_ADD_INTATTR, (a), (b)},
#define AGFW_ADD_STRATTR(a, b)     {AGFW_ADD_STRATTR, (a), (b)},
#define AGFW_ADDTYPE_END()         {AGFW_ADDTYPE_END, NULL, NULL},


/*
 * String attributes can have environment variables. Unit test would
 * expand them to the correspnding values. Following macro can be used
 * as the delimiter for the env vars.
 *
 * ex: AGFW_ADD_STRATTR("configfilePath", * "%CRS_HOME%/config/x.config");
 */
#define AGFW_ENV_SEPARATOR        '%'


/*
 * Adds a resource to the agfw unit test.
 *
 * Args:
 * AGFW_ADDRES_BEGIN(a, b) a = typename, b = resource name
 *
 * AGFW_MODIFY_INTATTR(a, b) a = attrname, b = integer attr value in
 * string format.
 *
 * AGFW_MODIFY_STRATTR(a, b) a = attrname, b = attr value in str
 * format.
 *
 * AGFW_ADDRES_END()  End marker for resource creation.
 *
 */
#define AGFW_ADDRES_BEGIN(a, b)        {AGFW_ADDRES_BEGIN, (a), (b)},
#define AGFW_MODIFY_INTATTR(a, b)      {AGFW_MODIFY_INTATTR, (a), (b)},
#define AGFW_MODIFY_STRATTR(a, b)      {AGFW_MODIFY_STRATTR, (a), (b)},
#define AGFW_ADDRES_END()              {AGFW_ADDRES_END, NULL, NULL},


/*
 * Adds a test operation.
 * args: a = resource name
 */
#define AGFW_START_RESOURCE(a)  {AGFW_START_RESOURCE, (a), NULL},
#define AGFW_STOP_RESOURCE(a)   {AGFW_STOP_RESOURCE, (a), NULL},
#define AGFW_CLEAN_RESOURCE(a)  {AGFW_CLEAN_RESOURCE, (a), NULL},
#define AGFW_FAULT_RESOURCE(a)  {AGFW_FAULT_RESOURCE, (a), NULL},
#define AGFW_SLEEP_SECS(a)      {AGFW_SLEEP_SECS, (a), NULL},
#define AGFW_CHECK_THRDS(a)     {AGFW_CHECK_THRDS, (a), NULL},
#define AGFW_TIMER_TEST(a)      {AGFW_TIMER_TEST, NULL, NULL},

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* CLSAGFW_MACROS */

